/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import io.quarkus.gradle.tasks.QuarkusBuild
import io.quarkus.gradle.tasks.QuarkusRun

plugins {
  alias(libs.plugins.quarkus)
  id("org.kordamp.gradle.jandex")
  id("polaris-runtime")
  // id("polaris-license-report")
}

val quarkusRunner by
  configurations.creating { description = "Used to reference the generated runner-jar" }

// Configuration to expose distribution artifacts
val distributionElements by
  configurations.creating {
    isCanBeConsumed = true
    isCanBeResolved = false
  }

dependencies {
  implementation(project(":polaris-runtime-service"))

  runtimeOnly(project(":polaris-eclipselink"))
  runtimeOnly("org.postgresql:postgresql")
  runtimeOnly(project(":polaris-relational-jdbc"))
  runtimeOnly("io.quarkus:quarkus-jdbc-postgresql")
  runtimeOnly(project(":polaris-extensions-federation-hadoop"))

  if ((project.findProperty("NonRESTCatalogs") as String?)?.contains("HIVE") == true) {
    runtimeOnly(project(":polaris-extensions-federation-hive"))
  }

  // enforce the Quarkus _platform_ here, to get a consistent and validated set of dependencies
  implementation(enforcedPlatform(libs.quarkus.bom))
  implementation("io.quarkus:quarkus-container-image-docker")
}

quarkus {
  quarkusBuildProperties.put("quarkus.package.type", "fast-jar")
  // Pull manifest attributes from the "main" `jar` task to get the
  // release-information into the jars generated by Quarkus.
  quarkusBuildProperties.putAll(
    provider {
      tasks
        .named("jar", Jar::class.java)
        .get()
        .manifest
        .attributes
        .map { e -> "quarkus.package.jar.manifest.attributes.\"${e.key}\"" to e.value.toString() }
        .toMap()
    }
  )
}

tasks.register("run") { dependsOn("quarkusRun") }

tasks.named<QuarkusRun>("quarkusRun") {
  jvmArgs =
    listOf(
      "-Dpolaris.bootstrap.credentials=POLARIS,root,s3cr3t",
      "-Dquarkus.console.color=true",
      "-Dpolaris.features.\"ALLOW_INSECURE_STORAGE_TYPES\"=true",
      "-Dpolaris.features.\"SUPPORTED_CATALOG_STORAGE_TYPES\"=[\"FILE\",\"S3\",\"GCS\",\"AZURE\"]",
      "-Dpolaris.readiness.ignore-severe-issues=true",
      "-Dpolaris.features.\"DROP_WITH_PURGE_ENABLED\"=true",
    )
}

val quarkusBuild = tasks.named<QuarkusBuild>("quarkusBuild")

// Expose runnable jar via quarkusRunner configuration for integration-tests that require the
// server.
artifacts {
  add(quarkusRunner.name, provider { quarkusBuild.get().fastJar.resolve("quarkus-run.jar") }) {
    builtBy(quarkusBuild)
  }
  add("distributionElements", layout.buildDirectory.dir("quarkus-app")) { builtBy("quarkusBuild") }
}
